package com.ruoyi.system.controller.activity;

import com.ruoyi.common.annotation.Log;
import com.ruoyi.common.annotation.RepeatSubmit;
import com.ruoyi.common.core.controller.BaseController;
import com.ruoyi.common.core.domain.AjaxResult;
import com.ruoyi.common.core.page.TableDataInfo;
import com.ruoyi.common.core.validate.AddGroup;
import com.ruoyi.common.core.validate.EditGroup;
import com.ruoyi.common.core.validate.QueryGroup;
import com.ruoyi.common.enums.BusinessType;
import com.ruoyi.common.utils.poi.ExcelUtil;
import com.ruoyi.system.domain.bo.ZkActivityObjBo;
import com.ruoyi.system.domain.vo.ZkActivityObjVo;
import com.ruoyi.system.service.IZkActivityObjService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.util.Arrays;
import java.util.List;

/**
 * 活动主体中的活动对象Controller
 *
 * @author ruoyi
 * @date 2021-10-27
 */
@Validated
//@Api(value = "活动主体中的活动对象控制器", tags = {"活动主体中的活动对象管理"})
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@RestController
@RequestMapping("/system/activityObj")
public class ZkActivityObjController extends BaseController {

    private final IZkActivityObjService iZkActivityObjService;

    /**
     * 查询活动主体中的活动对象列表
     */
//    @ApiOperation("查询活动主体中的活动对象列表")
    @PreAuthorize("@ss.hasPermi('system:activityObj:list')")
    @GetMapping("/list")
    public TableDataInfo<ZkActivityObjVo> list(@Validated(QueryGroup.class) ZkActivityObjBo bo) {
        return iZkActivityObjService.queryPageList(bo);
    }

    /**
     * 导出活动主体中的活动对象列表
     */
//    @ApiOperation("导出活动主体中的活动对象列表")
    @PreAuthorize("@ss.hasPermi('system:activityObj:export')")
    @Log(title = "活动主体中的活动对象", businessType = BusinessType.EXPORT)
    @GetMapping("/export")
    public void export(@Validated ZkActivityObjBo bo, HttpServletResponse response) {
        List<ZkActivityObjVo> list = iZkActivityObjService.queryList(bo);
        ExcelUtil.exportExcel(list, "活动主体中的活动对象", ZkActivityObjVo.class, response);
    }

    /**
     * 获取活动主体中的活动对象详细信息
     */
//    @ApiOperation("获取活动主体中的活动对象详细信息")
    @PreAuthorize("@ss.hasPermi('system:activityObj:query')")
    @GetMapping("/{id}")
    public AjaxResult<ZkActivityObjVo> getInfo(@NotNull(message = "主键不能为空")
                                               @PathVariable("id") Long id) {
        return AjaxResult.success(iZkActivityObjService.queryById(id));
    }

    /**
     * 新增活动主体中的活动对象
     */
//    @ApiOperation("新增活动主体中的活动对象")
    @PreAuthorize("@ss.hasPermi('system:activityObj:add')")
    @Log(title = "活动主体中的活动对象", businessType = BusinessType.INSERT)
    @RepeatSubmit()
    @PostMapping()
    public AjaxResult<Void> add(@Validated(AddGroup.class) @RequestBody ZkActivityObjBo bo) {
        return toAjax(iZkActivityObjService.insertByBo(bo) ? 1 : 0);
    }

    /**
     * 修改活动主体中的活动对象
     */
//    @ApiOperation("修改活动主体中的活动对象")
    @PreAuthorize("@ss.hasPermi('system:activityObj:edit')")
    @Log(title = "活动主体中的活动对象", businessType = BusinessType.UPDATE)
    @RepeatSubmit()
    @PutMapping()
    public AjaxResult<Void> edit(@Validated(EditGroup.class) @RequestBody ZkActivityObjBo bo) {
        return toAjax(iZkActivityObjService.updateByBo(bo) ? 1 : 0);
    }

    /**
     * 删除活动主体中的活动对象
     */
//    @ApiOperation("删除活动主体中的活动对象")
    @PreAuthorize("@ss.hasPermi('system:activityObj:remove')")
    @Log(title = "活动主体中的活动对象", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public AjaxResult<Void> remove(@NotEmpty(message = "主键不能为空")
                                   @PathVariable Long[] ids) {
        return toAjax(iZkActivityObjService.deleteWithValidByIds(Arrays.asList(ids), true) ? 1 : 0);
    }
}
