package com.gaowj.business.util;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.commons.lang.StringUtils;

import com.gaowj.business.exception.BusinessException;

/**
 * 数据持久层工具类
 * 
 * @author luosz@jfbrother.com
 *
 */
public class DaoUtil {
	/**
	 * 新增或者删除时，构造的字段list(key-value映射)
	 */
	public static final String INFO = "infos";
	/**
	 * 映射中key的名称
	 */
	private static final String KEY = "key";
	/**
	 * 映射中value的名称
	 */
	private static final String VALUE = "value";

	/**
	 * 添加新建信息（UUID,CREATE_ID,CREATE_TIME,UPDATE_ID,UPDATE_TIME）
	 * 
	 * @param entity
	 *            填充的对象
	 * @param ignoreField
	 *            需要忽略的字段
	 * @date 2018-09-01 下午5:25:27
	 * @author luosz@jfbrother.com
	 */
	public static void addCreateInfo(Map<String, Object> entity, String... ignoreFields) {
		// 构造模板
		Map<String, Object> entityModel = new HashMap<String, Object>();
		entityModel.put("UUID", java.util.UUID.randomUUID().toString());
		entityModel.put("CREATE_ID", SessionUtil.getCode());
		entityModel.put("CREATE_TIME", new Date());
		entityModel.put("UPDATE_ID", SessionUtil.getCode());
		entityModel.put("UPDATE_TIME", new Date());
		addCommonInfo(entity, entityModel, ignoreFields);
	}

	/**
	 * 添加更新信息（UPDATE_ID,UPDATE_TIME）
	 * 
	 * @param entity
	 *            填充的对象
	 * @param ignoreFields
	 *            需要忽略的字段
	 * @date 2018-09-01 下午7:32:48
	 * @author luosz@jfbrother.com
	 */
	public static void addUpdateInfo(Map<String, Object> entity, String... ignoreFields) {
		// 构造模板
		Map<String, Object> entityModel = new HashMap<String, Object>();
		entityModel.put("UPDATE_ID", SessionUtil.getCode());
		entityModel.put("UPDATE_TIME", new Date());
		addCommonInfo(entity, entityModel, ignoreFields);
	}

	/**
	 * 对entity对象添加entityModel中的元素，但是过滤掉ignoreFields中指定的key(即不覆盖ignoreFields指定的key值)
	 * 
	 * @param entity
	 *            填充的对象
	 * @param entityModel
	 *            新值
	 * @param ignoreFields
	 *            需要忽略的字段
	 * @date 2018-09-05 下午1:44:45
	 * @author luosz@jfbrother.com
	 */
	public static void addCommonInfo(Map<String, Object> entity, Map<String, Object> entityModel, String... ignoreFields) {
		// 剔除忽略的属性
		if (ignoreFields != null) {
			for (String ignoreField : ignoreFields) {
				entityModel.remove(ignoreField);
			}
		}
		// 将模板中剩余的属性覆盖到entity
		if (entityModel.keySet() != null) {
			for (Entry<String, Object> field : entityModel.entrySet()) {
				entity.put(field.getKey(), field.getValue());
			}
		}
	}

	/**
	 * 将entity中的字段信息整理成key-value映射的列表，并放入info属性
	 * 
	 * @param entity
	 *            需要构造的对象
	 * @param ignoreFields
	 *            构造中需要忽略的字段
	 * @date 2018-09-01 下午6:01:07
	 * @author luosz@jfbrother.com
	 */
	public static void addFieldsInfo(Map<String, Object> entity, String... ignoreFields) throws BusinessException {
		addFieldsInfo(INFO, KEY, VALUE, entity, ignoreFields);
	}

	/**
	 * 将entity中的字段信息整理成key-value映射的列表，并放入info属性（可自由指定增加的列表的key值和key-value对应的key值）
	 * 
	 * @param infoName
	 *            列表的key值
	 * @param keyName
	 *            key元素的key值
	 * @param valueName
	 *            value元素的key值
	 * @param entity
	 *            需要构造的对象
	 * @param ignoreFields
	 *            构造中需要忽略的字段
	 * @throws BusinessException
	 * @date 2018-09-05 下午1:50:05
	 * @author luosz@jfbrother.com
	 */
	public static void addFieldsInfo(String infoName, String keyName, String valueName, Map<String, Object> entity, String... ignoreFields) throws BusinessException {
		if (entity.keySet() != null) {
			// 字段列表
			List<Map<String, Object>> listField = new ArrayList<Map<String, Object>>();

			// 将map集合中的key和value 取出来分别放到list集合里
			outer: for (String str : entity.keySet()) {
				if (ignoreFields != null) {
					for (String ignoreField : ignoreFields) {
						// 如果该字段被忽略，则继续外层循环
						if (StringUtils.equalsIgnoreCase(ignoreField, str))
							continue outer;
					}
				}

				// 构造key-value映射对象
				Map<String, Object> mapping = new HashMap<String, Object>();
				mapping.put(keyName, str);
				mapping.put(valueName, entity.get(str));

				listField.add(mapping);
			}
			if (listField.size() > 0) {
				entity.put(infoName, listField);
				return;
			}
		}
		throw new BusinessException(infoName + "构造失败！");
	}
}
